<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MapLocation;
use App\Models\MapSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class MapEditorController extends Controller
{
    /**
     * Display the map editor page.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $mapSettings = MapSetting::first();
        
        return view('admin.map-editor', [
            'mapSettings' => $mapSettings,
        ]);
    }

    /**
     * Update map settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'map_title' => 'required|string|max:255',
            'map_description' => 'nullable|string',
            'map_image' => 'nullable|image|max:2048',
        ]);

        $mapSettings = MapSetting::first();
        if (!$mapSettings) {
            $mapSettings = new MapSetting();
        }

        $mapSettings->title = $request->map_title;
        $mapSettings->description = $request->map_description;

        // Handle image upload
        if ($request->hasFile('map_image')) {
            // Delete old image if exists
            if ($mapSettings->image_path && Storage::exists('public/' . $mapSettings->image_path)) {
                Storage::delete('public/' . $mapSettings->image_path);
            }

            // Store new image
            $imagePath = $request->file('map_image')->store('maps', 'public');
            $mapSettings->image_path = $imagePath;
        }

        $mapSettings->save();

        return redirect()->route('admin.map.editor')->with('success', 'Map settings updated successfully.');
    }

    /**
     * Get all map locations.
     *
     * @return \Illuminate\Http\Response
     */
    public function getLocations()
    {
        $locations = MapLocation::all();

        return response()->json($locations);
    }

    /**
     * Create a new location.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
public function storeLocation(Request $request)
{
    \Log::info('Location store attempt', $request->all());
    
    try {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string|max:30',
            'description' => 'required|string',
            'x' => 'required|numeric|between:0,100',
            'y' => 'required|numeric|between:0,100',
            'details' => 'nullable|array',
        ]);
        
        \Log::info('Validation passed', $validated);
        
        $location = MapLocation::create($validated);
        
        \Log::info('Location created', ['id' => $location->id]);
        
        return response()->json($location, 201);
    } catch (\Exception $e) {
        \Log::error('Error creating location', [
            'message' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json(['error' => $e->getMessage()], 500);
    }
}

    /**
     * Update an existing location.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateLocation(Request $request, $id)
    {
        $location = MapLocation::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string|max:30',
            'description' => 'required|string',
            'x' => 'required|numeric|between:0,100',
            'y' => 'required|numeric|between:0,100',
            'details' => 'nullable|array',
        ]);

        $location->update($validated);

        return response()->json($location);
    }

    /**
     * Delete a location.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function deleteLocation($id)
    {
        $location = MapLocation::findOrFail($id);
        $location->delete();

        return response()->json(['message' => 'Location deleted successfully.']);
    }

    /**
     * Delete multiple locations.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function deleteMultipleLocations(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'integer|exists:map_locations,id',
        ]);

        MapLocation::whereIn('id', $request->ids)->delete();

        return response()->json(['message' => count($request->ids) . ' locations deleted successfully.']);
    }

    /**
     * Update location order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateOrder(Request $request)
    {
        $request->validate([
            'order' => 'required|array',
            'order.*' => 'integer|exists:map_locations,id',
        ]);

        foreach ($request->order as $index => $id) {
            MapLocation::where('id', $id)->update(['order' => $index]);
        }

        return response()->json(['message' => 'Order updated successfully.']);
    }
}